function [varargout] = SCM(varargin)

if nargin == 0 %LAUNCH GUI
    
    % Declare all global variables.
    global data comp_str name_u
    
    % LAYOUT
    data = [];
    data.SoftwareVersion = '2019-03-1';
    
    data.Load.Radius = [];
    data.Load.Length = [];
    data.Load.Offset = [];
    
    data.Load.Angle = []; % one angle btw alpha and beta
    data.popupmenu_angle = [];
    
    data.Load.Omega = [];
    
    data.Load.P_Diam = [];
    data.Load.R_Mass = [];
    data.Load.L_Mass = [];
    data.Load.P_Mass = [];
    data.Load.R_k = [];
    data.Load.L_k = [];
    data.Load.J1 = [];
    data.Load.J2 = [];
    data.Load.t_s = [];
    
    data.Load.PT = []; % known between force and torque
    data.popupmenu_pt = [];
    
    data.pushbutton_sketch = [];
    
    data.pushbutton_solve_dyna = [];
    data.pushbutton_plot1 = [];
    data.pushbutton_plot2 = [];
    
    data.popupmenu_plot1 = [];
    data.popupmenu_plot2 = [];
    
    % Cell array containing all objects nomenclature.
    comp_str = {'Element','Text';...
        'Radius','Crank radius ''r'' [mm]';...
        'Length','Rod length ''l'' [mm]';...
        'Offset','Offset ''f'' [mm]';...
        'Alpha','Alpha ''a'' [rad]';...
        'Beta','Beta ''b'' [rad]';...
        'Omega','a speed ''w'' [rad/s] CW';...
        'P_Diameter','Piston diameter ''d'' [mm]';...
        'P_Mass','Piston mass ''m'' [Kg]';...
        'R_Mass','Crank mass ''mr'' [Kg]';...
        'L_Mass','Rod mass ''ml'' [Kg]';...
        'Pressure','Piston pressure ''P'' [MPa]';...
        'Torque','Torque ''T'' [Nm]';...
        'R_k','COG(R) param. ''kr'' [mm]';...
        'L_k','COG(L) param. ''kl'' [mm]';...
        'J1','R Inertia ''J1'' [Kg*mm^2]';...
        'J2','L Inertia ''J2'' [Kg*mm^2]';...
        't','Time span ''t_s'' [s]'};
    
    name_u = {'x_G1','y_G1','\alpha','x_G2','y_G2','\beta','y_P', ...
    'xd_G1','yd_G1','\alpha d','xd_G2','yd_G2','\beta d','yd_P', ...
    'R_Ox','R_Oy','R_Ax','R_Ay','R_Bx','R_By'; ...
    '[mm]','[mm]','[rad]','[mm]','[mm]','[rad]','[mm]',...
    '[mm/s]','[mm/s]','[rad/s]','[mm/s]','[mm/s]','[rad/s]','[mm/s]',...
    '[N]','[N]','[N]','[N]','[N]','[N]'};
    
    % Default layout values
    elem_b = [0.03 0.04 0.095 0.20 0.41 0.515 0.73];
    elem_h = [0.08 0.06 0.045];
    matrix_b = [0.035 0.14 0.245 0.35 0.455 0.56 0.665 0.77 0.875];
    matrix_h = 0.03+elem_h(1,1)*[11 10 9 8 7 6 5 4 3 2 1 0];
    
    appdata = [];
    appdata.GUIDEOptions = struct(...
        'active_h', [], ...
        'taginfo', struct(...
        'togglebutton', 2), ...
        'override', 0, ...
        'release', 12, ...
        'resize', 'simple', ...
        'accessibility', 'callback', ...
        'mfile', 0, ...
        'callbacks', 1, ...
        'singleton', 1, ...
        'syscolorfig', 1, ...
        'blocking', 0);
    
    % Create the figure fig = figure(...)
    appdata.lastValidTag = '';
    appdata.GUIDELayoutEditor = [];
    fig = figure(...
        'MenuBar','none',...
        'PaperUnits',get(0,'defaultfigurePaperUnits'),...
        'Color',[1 1 1],...
        'Colormap',get(0,'defaultfigureColormap'),...
        'IntegerHandle','off',...
        'InvertHardcopy',get(0,'defaultfigureInvertHardcopy'),...
        'Name','Control Panel',...
        'NumberTitle','off',...
        'PaperPosition',[0.6345175 6.345175 20.30456 15.22842],... % Position on printable paper
        'PaperSize',[20.98404194812 29.67743169791],... % Size of paper (mm)
        'PaperType',get(0,'defaultfigurePaperType'),...
        'Position',[1240 350 640 450],... % 'OuterPosition',[0 1080-400 550 400],... %position on screen [x y b h] da angolo in basso a sinistra
        'HandleVisibility','callback',...
        'Tag','Control Panel',...
        'UserData',[7.00439453125;11.00439453125;15.00439453125;159.00341796875;164.00341796875;166.003295898438;168.003295898438;170.003295898438;172.003295898438;174.003295898438;180.003295898438;178.003295898438;182.003295898438;185.003295898438;187.003295898438;188.003295898438;190.003295898438],...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Frame
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[0.02 0.02 0.96 0.96],...
        'Style','frame',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Internal grid division
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'Position',[0.02 0.02+1*elem_h(1,1) 0.96 10*elem_h(1,1)], ...
        'Style','frame', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'Position',[0.02 0.02+1*elem_h(1,1) 0.96 9*elem_h(1,1)], ...
        'Style','frame', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'Position',[0.02 0.02+1*elem_h(1,1) 0.96 7*elem_h(1,1)], ...
        'Style','frame', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'Position',[0.02 0.02+1*elem_h(1,1) 0.96 6*elem_h(1,1)], ...
        'Style','frame', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'Position',[0.02 0.02+1*elem_h(1,1) 0.96 5*elem_h(1,1)], ...
        'Style','frame', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Title
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'FontSize',12,...
        'FontWeight','demi',...
        'Position',[matrix_b(1,1)+0.04 0.905 0.86 elem_h(1,2)],...
        'String','Slider-Crank Mechanism',...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Visualize the data
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'FontWeight','demi', ...
        'Position',[0.75 0.91 0.22 elem_h(1,2)], ...
        'String',['ver. ' data.SoftwareVersion], ...
        'Style','text', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    %% Sketch Tab
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'FontSize',9, ...
        'FontWeight','demi', ...
        'Position',[matrix_b(1,1) matrix_h(1,2) elem_b(1,5) elem_h(1,3)], ...
        'horizontalAlignment','left',...
        'String','Visualize System Sketch', ...
        'Style','text', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.pushbutton_sketch = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,2) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', @sketch_plot, ... % Callback 'sketch_plot' function
        'String','Sketch',...
        'Style','pushbutton',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    set(data.pushbutton_sketch, 'Enable', 'on')
    
    %% Geometry Tab
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'FontSize',9, ...
        'FontWeight','demi', ...
        'Position',[matrix_b(1,1) matrix_h(1,3) elem_b(1,3) elem_h(1,3)], ...
        'horizontalAlignment','left',...
        'String','Geometry', ...
        'Style','text', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Crank
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,3) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(2,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.Radius = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,3) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'r'}, ...
        'String','43.65',...
        'Style','edit',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Connecting Rod
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,6) matrix_h(1,3) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(3,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.Length = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,3) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum, 'l'}, ...
        'String','145',...
        'Style','edit',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Offset
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,4) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(4,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.Offset = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,4) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'f'}, ...
        'String','0.5',...
        'Style','edit',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    %% Kinematics Tab
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'FontSize',9, ...
        'FontWeight','demi', ...
        'Position',[matrix_b(1,1) matrix_h(1,5) elem_b(1,3) elem_h(1,3)], ...
        'horizontalAlignment','left',...
        'String','Kinematics', ...
        'Style','text', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Angle
    appdata = [];
    appdata.lastValidTag = '';
    data.popupmenu_angle = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,5) elem_b(1,4) elem_h(1,3)],...
        'String',{char(comp_str(5,2));char(comp_str(6,2))},...
        'Style','popupmenu',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.Angle = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,5) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'ang'}, ...
        'String','0',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Angular Speed (Omega)
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,6) matrix_h(1,5) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(7,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.Omega = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,5) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'w'}, ...
        'String','0',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    %% Kinematic Plot Tab
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'FontSize',9, ...
        'FontWeight','demi', ...
        'Position',[matrix_b(1,1) matrix_h(1,6) elem_b(1,4) elem_h(1,3)], ...
        'horizontalAlignment','left',...
        'String','Kinematic Plots', ...
        'Style','text', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.popupmenu_plot1 = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,6) elem_b(1,4)+elem_b(1,3) elem_h(1,2)],...
        'Style','popupmenu',...
        'String',{'Piston Displacement vs Time','Piston Velocity vs Time','Piston Acceleration vs Time'},...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.pushbutton_plot1 = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,6) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', @sub_plot1, ... % Callback 'sub_plot' function
        'String','Plot',...
        'Style','pushbutton',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    %% Dynamics Tab
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'FontSize',9, ...
        'FontWeight','demi', ...
        'Position',[matrix_b(1,1) matrix_h(1,7) elem_b(1,3) elem_h(1,3)], ...
        'horizontalAlignment','left',...
        'String','Dynamics', ...
        'Style','text', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Piston Diameter
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,7) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(8,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.P_Diam = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,7) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'d'}, ...
        'String','81.40',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Piston Mass
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,8) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(9,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.P_Mass = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,8) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'mp'}, ...
        'String','0.546',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Crank Mass
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,9) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(10,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.R_Mass = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,9) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'mr'}, ...
        'String','3.043',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Rod Mass
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,10) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(11,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.L_Mass = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,10) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'ml'}, ...
        'String','0.879',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Force [Pressure & Torque]
    appdata = [];
    appdata.lastValidTag = '';
    data.popupmenu_pt = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,2) matrix_h(1,11) elem_b(1,4) elem_h(1,3)],...
        'String',{char(comp_str(12,2));char(comp_str(13,2))},...
        'Style','popupmenu',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.PT = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,11) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'pt'}, ...
        'String','0',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Crank k
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,6) matrix_h(1,7) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(14,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.R_k = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,7) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'kr'}, ...
        'String','-3.89',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Rod k
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,6) matrix_h(1,8) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(15,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.L_k = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,8) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'kl'}, ...
        'String','41.07',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Inertia moment of the Crank
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,6) matrix_h(1,9) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(16,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.J1 = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,9) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'j1'}, ...
        'String','5968.522',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Inertia moment of the Rod
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,6) matrix_h(1,10) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(17,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.J2 = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,10) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'j2'}, ...
        'String','3369.095',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    % Time span
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,6) matrix_h(1,11) elem_b(1,4) elem_h(1,3)],...
        'String',char(comp_str(18,2)),...
        'Style','text',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.Load.t_s = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,11) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', {@sub_inputNum,'t_s'}, ...
        'String','0:0.001:10',...
        'Style','edit',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    %% Dynamic Solve button
    
    appdata = [];
    appdata.lastValidTag = '';
    data.pushbutton_solve_dyna = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,12) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', @sub_solve_dyna, ... % Callback 'sub_solve_dyna' function
        'String','Solve Dynamic',...
        'Style','pushbutton',...
        'Enable','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    
    %% Dynamic Plot Tab
    
    appdata = [];
    appdata.lastValidTag = '';
    uicontrol( ...
        'Parent',fig, ...
        'Units','normalized', ...
        'BackgroundColor',[1 1 1], ...
        'FontSize',9, ...
        'FontWeight','demi', ...
        'Position',[matrix_b(1,1) matrix_h(1,12) elem_b(1,4) elem_h(1,3)], ...
        'horizontalAlignment','left',...
        'String','Dynamic Plots', ...
        'Style','text', ...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.popupmenu_plot2 = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,4)+elem_b(1,2) matrix_h(1,12) elem_b(1,4)+elem_b(1,3) elem_h(1,2)],...
        'horizontalAlignment','left',...
        'Style','popupmenu',...
        'String',{'x (G1)','y (G1)','a','x (G2)','y (G2)','b','y (P)', ...
                 'xd (G1)','yd (G1)','a_d','xd (G2)','yd (G2)','b_d','yd (P)', ...
                 'R (Ox)','R (Oy)','R (Ax)','R (Ay)','R (Bx)','R (By)','T/F vs Time','Animation'},...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    appdata = [];
    appdata.lastValidTag = '';
    data.pushbutton_plot2 = uicontrol(...
        'Parent',fig,...
        'Units','normalized',...
        'BackgroundColor',[1 1 1],...
        'Position',[matrix_b(1,7)+elem_b(1,3) matrix_h(1,12) elem_b(1,4)-elem_b(1,2) elem_h(1,2)],...
        'Callback', @sub_plot2, ... % Callback 'sub_plot2' function
        'String','Plot',...
        'Style','pushbutton',...
        'Enable','off',...
        'Visible','off',...
        'CreateFcn', {@local_CreateFcn, '', appdata} );
    
    
    %% Generate a structure of handles to pass to callbacks, and store it.
    
    handles = guihandles(fig);
    guidata(fig, handles);
    
    if nargout > 0
        varargout{1} = fig;
    end
    
    set(fig,'UserData',data);
    
elseif ischar(varargin{1}) % INVOKE NAMED SUBFUNCTION OR CALLBACK
    try
        [varargout{1:nargout}] = feval(varargin{:}); % FEVAL switchyard
    catch ME
        rethrow(ME);
    end
end

%% FUNCTION sub_inputNum
function [varargout] = sub_inputNum(~, ~, ~)

global data...
    r l f...
    ang a b w...
    d mp mr ml...
    kr kl...
    F T pt...
    j1 j2...
    t_s

r = [];
l = [];
f = [];

ang = [];
w = [];

d = [];
mp = [];
mr = [];
ml = [];

pt = [];
F = [];
T = [];

kr = [];
kl = [];

j1 = [];
j2 = [];

t_s = [];

%Get all values from the users

Geom_check = 1;
Kine_check = 1;
Dyna_check = 1;

try %r
    r = eval(get(data.Load.Radius,'string'))/1000;
end
try %l
    l = eval(get(data.Load.Length,'string'))/1000;
end
try %f
    f = eval(get(data.Load.Offset,'string'))/1000;
end
try %ang
    ang = eval(get(data.Load.Angle,'string'));
end
try %w
    w = eval(get(data.Load.Omega,'string'));
end
try %d
    d = eval(get(data.Load.P_Diam,'string'))/1000;
end
try %mp
    mp = eval(get(data.Load.P_Mass,'string'));
end
try %mr
    mr = eval(get(data.Load.R_Mass,'string'));
end
try %ml
    ml = eval(get(data.Load.L_Mass,'string'));
end
try %pt
    pt = eval(get(data.Load.PT,'string'));
catch
    pt = 'expression';
end
try %kr
    kr = eval(get(data.Load.R_k,'string'))/1000;
end
try %kl
    kl = eval(get(data.Load.L_k,'string'))/1000;
end
try %j1
    j1 = eval(get(data.Load.J1,'string'))/10^6;
end
try %j2
    j2 = eval(get(data.Load.J2,'string'))/10^6;
end
try %t_s
    t_s = eval(get(data.Load.t_s,'string'));
catch
    t_s = 'expression';
end

%angle popup menu
ang_sel = get(data.popupmenu_angle,'Value');
switch ang_sel
    case 1 %a
        a = ang;
        b = asin((r*sin(a)-f)/l);
    case 2 %b
        b = ang;
        a = asin((l*sin(b)+f)/r);
    otherwise
end

%Control on single values. Get red if they are not admissible or not scalar
r_p = 0.5; %r_p=0 -> pure red / r_p=1 -> pure white

if isempty(r) == 1 || r <= 0
        set(data.Load.Radius,'BackgroundColor',[1 r_p r_p]);
        set(data.Load.Radius,'string','43.65');
        r = 0;
        Geom_check = 0;
    else
        set(data.Load.Radius,'BackgroundColor',[1 1 1]);
end

if isempty(l) == 1 || l <= 0 || l < r
    set(data.Load.Length,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.Length,'string','145');
    l = 0;
    Geom_check = 0;
else
    set(data.Load.Length,'BackgroundColor',[1 1 1]);
end

if isempty(f) == 1 || f > abs(r - l)
    set(data.Load.Offset,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.Offset,'string','0.5');
    Geom_check = 0;
else
    set(data.Load.Offset,'BackgroundColor',[1 1 1]);
end

%Enable Kinematic tab if all Geom. values are ok
if Geom_check == 1
    set(data.Load.Angle,'Enable','on');
    set(data.Load.Omega,'Enable','on');
else
    set(data.Load.Angle,'Enable','off');
    set(data.Load.Omega,'Enable','off');
end

%Control on single values. Get red if they are not admissible nor scalar
if isempty(ang) == 1
    set(data.Load.Angle,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.Angle,'string','0');
    ang = 0;
    Kine_check = 0;
else
    set(data.Load.Angle,'BackgroundColor',[1 1 1]);
end

if isempty(w) == 1
    set(data.Load.Omega,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.Omega,'string','0');
    w = 0;
    Kine_check = 0;
else
    set(data.Load.Omega,'BackgroundColor',[1 1 1]);
end

%Enable Plot1 button if all Kine values are ok
if Geom_check == 1 && Kine_check == 1
    set(data.pushbutton_plot1,'Enable','on');
else
    set(data.pushbutton_plot1,'Enable','off');
end

%Enable Dynamics tab if all Kine. values are ok
if Geom_check == 1 && Kine_check == 1
    set(data.Load.P_Diam,'Enable','on');
    set(data.Load.P_Mass,'Enable','on');
    set(data.Load.R_Mass,'Enable','on');
    set(data.Load.L_Mass,'Enable','on');
    set(data.Load.PT,'Enable','on');
    set(data.Load.R_k,'Enable','on');
    set(data.Load.L_k,'Enable','on');
    set(data.Load.J1,'Enable','on');
    set(data.Load.J2,'Enable','on');
    set(data.Load.t_s,'Enable','on');
else
    set(data.Load.P_Diam,'Enable','off');
    set(data.Load.P_Mass,'Enable','off');
    set(data.Load.R_Mass,'Enable','off');
    set(data.Load.L_Mass,'Enable','off');
    set(data.Load.PT,'Enable','off');
    set(data.Load.R_k,'Enable','off');
    set(data.Load.L_k,'Enable','off');
    set(data.Load.J1,'Enable','off');
    set(data.Load.J2,'Enable','off');
    set(data.Load.t_s,'Enable','off');
end

%Control on single values. Get red if they are not admissible nor scalar
if isempty(d) == 1 || d == 0
    set(data.Load.P_Diam,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.P_Diam,'string','81.40');
    Dyna_check = 0;
else
    set(data.Load.P_Diam,'BackgroundColor',[1 1 1]);
end

if isempty(mp) == 1 || mp == 0
    set(data.Load.P_Mass,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.P_Mass,'string','0.546');
    Dyna_check = 0;
else
    set(data.Load.P_Mass,'BackgroundColor',[1 1 1]);
end

if isempty(mr) == 1 || mr == 0
    set(data.Load.R_Mass,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.R_Mass,'string','3.043');
    Dyna_check = 0;
else
    set(data.Load.R_Mass,'BackgroundColor',[1 1 1]);
end

if isempty(ml) == 1 || ml == 0
    set(data.Load.L_Mass,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.L_Mass,'string','0.879');
    Dyna_check = 0;
else
    set(data.Load.L_Mass,'BackgroundColor',[1 1 1]);
end

if isempty(pt) == 1
    set(data.Load.PT,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.PT,'string','0');
    Dyna_check = 0;
else    
    set(data.Load.PT,'BackgroundColor',[1 1 1]);
end

if isempty(kr) == 1
    set(data.Load.R_k,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.R_k,'string','-3.89');
    Dyna_check = 0;
else
    set(data.Load.R_k,'BackgroundColor',[1 1 1]);
end

if isempty(kl) == 1
    set(data.Load.L_k,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.L_k,'string','41.07');
    Dyna_check = 0;
else
    set(data.Load.L_k,'BackgroundColor',[1 1 1]);
end

if isempty(j1) == 1 || j1 == 0
    set(data.Load.J1,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.J1,'string','5968.522');
    Dyna_check = 0;
else
    set(data.Load.J1,'BackgroundColor',[1 1 1]);
end

if isempty(j2) == 1 || j2 == 0
    set(data.Load.J2,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.J2,'string','3369.095');
    Dyna_check = 0;
else
    set(data.Load.J2,'BackgroundColor',[1 1 1]);
end

if isempty(t_s) == 1
    set(data.Load.t_s,'BackgroundColor',[1 r_p r_p]);
    set(data.Load.t_s,'string','0:0.1:10');
    Dyna_check = 0;
else    
    set(data.Load.PT,'BackgroundColor',[1 1 1]);
end
%Enable Plot2 button if all values are ok

set(data.pushbutton_plot2,'Enable','off');
set(data.pushbutton_plot2,'Visible','off');
set(data.pushbutton_solve_dyna,'Visible','on');

if Geom_check == 1 && Kine_check == 1 && Dyna_check == 1
    set(data.pushbutton_solve_dyna,'Enable','on');
else
    set(data.pushbutton_solve_dyna,'Enable','off');
end

%% FUNCTION sketck_plot
function [varargout] = sketch_plot (varargin)

figure(1)
set(gcf, 'Color', [1 1 1]);
set(gcf, 'Name', 'System sketch');

a = 0.8;
r = 80;
l = 145;
f = 20;
b = asin((r*sin(a)-f)/l);
k1 = -3.89;
k2 = 41.07;

p_b = 50; %piston base
p_h = 25; %piston height

x_crank = [0 r*sin(a)];
y_crank = [0 r*cos(a)];
x_length = [r*sin(a) r*sin(a)-l*sin(b)];
y_length = [r*cos(a) r*cos(a)+l*cos(b)];
x_G1 = k1*sin(a);
y_G1 = k1*cos(a);
x_G2 = r*sin(a)-k2*sin(b);
y_G2 = r*cos(a)+k2*cos(b);

plot(x_crank, y_crank,'r-',...
    x_length, y_length,'b-',...
    x_G1, y_G1, 'og',...
    x_G2, y_G2, 'og',...
    'LineWidth',3)

txt1 = ' \leftarrow R = Crank';
text(r*sin(a)/2,r*cos(a)/2,txt1,...
    'HorizontalAlignment','left')
txt2 = ' \leftarrow L = Rod';
text(r*sin(a)-l*sin(b)/2,r*cos(a)+l*cos(b)/2,txt2,...
    'HorizontalAlignment','left')

txt2 = 'G1';
text(k1*sin(a), k1*cos(a), txt2,...
    'VerticalAlignment','bottom','HorizontalAlignment','right')
txt3 = 'G2';
text(r*sin(a)-k2*sin(b), r*cos(a)+k2*cos(b), txt3,...
    'VerticalAlignment','top','HorizontalAlignment','right')

hold on

x_piston = [r*sin(a)-l*sin(b)-p_b/2 r*sin(a)-l*sin(b)+p_b/2 r*sin(a)-l*sin(b)+p_b/2 r*sin(a)-l*sin(b)-p_b/2 r*sin(a)-l*sin(b)-p_b/2];
y_piston = [r*cos(a)+l*cos(b)-p_h/2 r*cos(a)+l*cos(b)-p_h/2 r*cos(a)+l*cos(b)+p_h/2 r*cos(a)+l*cos(b)+p_h/2 r*cos(a)+l*cos(b)-p_h/2];
plot(x_piston, y_piston,'b-','LineWidth',1,'Handlevisibility','on')

y_ax = [0 r*cos(a)+l*cos(b)];
x_ax = [0 0];
plot(x_ax, y_ax,'k:','LineWidth',1,'Handlevisibility','on')

y_f = [(r*cos(a)+l*cos(b))/2 r*cos(a)+l*cos(b)];
x_f = [f f];
plot(x_f, y_f,'k:','LineWidth',1,'Handlevisibility','on')
text(f/2, r*cos(a)+l*cos(b)/2, 'F = offset','HorizontalAlignment','center')

radius = 0:0.05:2;
plot(r*cos(radius), r*sin(radius),'k:','LineWidth',1,'Handlevisibility','on')

text(x_length, y_length,{' A',' B'},'HorizontalAlignment','left')
text(0, 0, ' O','HorizontalAlignment','left')

alpha = a:0.05:pi/2;
plot(20*cos(alpha), 20*sin(alpha),'k:','LineWidth',1,'Handlevisibility','on')

beta = 0:0.05:b;
plot(r*sin(a)-l*sin(b)+40*sin(beta), r*cos(a)+l*cos(b)-40*cos(beta),'k:','LineWidth',1,'Handlevisibility','on')

axis off

text(r*sin(a)/7, 20,'\alpha','HorizontalAlignment','right','VerticalAlignment','top')
text(f+1, r*cos(a)+l*cos(b)-27,'\beta','VerticalAlignment','top')

%xlim([-r*cos(a)-l*cos(b)-l/10 l/10])
%ylim([0 r*sin(a)+r/10])

axis equal

%% FUNCTION sub_plot1 Kinematic
function [varargout] = sub_plot1 (varargin)

global data...
    r l f...
    a w...
    
t = 0:0.01:8*pi; % [s]
a_t = (w.*t)+a; % a(t) crank angle motion equation
b_t = asin((r.*sin(a_t)-f)/l);

yp = r.*cos(a_t) + l.*cos(b_t); % yp(t) y-displacement of B (piston)

TDC = yp.*0 + max(yp); % Top Dead Center
BDC = yp.*0 + min(yp); % Bottom Dead Center

vyp = -w*r.*(sin(a_t)+(r.*sin(a_t)-f).*cos(a_t)./(l^2-(r*sin(a_t)-f).^2).^0.5); % y.p(t) y-velocity of B (piston)

%%% frequent polynomial
Q = (((l/r)^2)-((sin(a_t)-(f/r)).^2)).^0.5;
%%%

Q1 = l^2.*(1-((r*sin(a_t)-f).^2)/(l^2)).^(1/2);
%ayp = (w^(2)*sin(a_t).*(r*sin(a_t)-f))/(Q) - (w^(2)*cos(a_t).^(2))/(Q) - (w^(2)*cos(a_t).^(2).*(r*sin(a_t)-f).^(2))/(r*Q.^3) - r*cos(a_t).*w^2; % y:p(t) y-acceleration of B (piston)
ayp = l*(- ((r.*w.*cos(a_t)).^2./(Q1)) + ((r.*w.^2.*sin(a_t).*(r*sin(a_t)-f))/(Q1)) - ((r.*w.*cos(a_t).*(r*sin(a_t)-f)).^2/(l.^4.*(1-((r*sin(a_t)-f).^2)/(l.^2)).^(3/2)))) - r.*w.^2.*cos(a_t);

var = get(data.popupmenu_plot1,'Value');
switch var
    case 1 %yp vs t (Piston displacement vs time)
        figure(2)
        set(gcf, 'Color', [1 1 1]);
        set(gcf, 'Name', 'Piston Displacement vs Time');
        
        plot(t, yp,...
            '-k','LineWidth',2,...
            'Displayname','\itposition')
        hold on
        
        plot(t, TDC,...
            ':k','LineWidth',1,...
            'Displayname','\itTDC')
        plot(t, BDC,...
            ':k','LineWidth',1,...
            'Displayname','\itBDC')
        
        xlabel('Time [s]', 'Fontsize', 12)
        ylabel('D - displacement [mm]', 'Fontsize', 12)
        xlim([0 6*pi])
        %plot(xlim,0*[1 1], ':k', 'Handlevisibility', 'off')
        legend('show')
        
    case 2 %y.p vs t (Piston velocity vs time)
        figure(3)
        set(gcf, 'Color', [1 1 1]);
        set(gcf, 'Name', 'Piston Velocity vs Time');
        
        plot(t, vyp,...
            '-k','LineWidth',2,...
            'Displayname','\itvelocity')
        hold on
        
        xlabel('Time [s]', 'Fontsize', 12)
        ylabel('V - velocity [mm/s]', 'Fontsize', 12)
        xlim([0 6*pi])
        plot(xlim,0*[1 1], ':k', 'Handlevisibility', 'off')
        legend('show')
        
    case 3 %y:p vs t (Piston acceleration vs time)
        figure(4)
        set(gcf, 'Color', [1 1 1]);
        set(gcf, 'Name', 'Piston Acceleration vs Time');
        
        plot(t, ayp,...
            '-k','LineWidth',2,...
            'Displayname','\itacceleration')
        hold on
        
        xlabel('Time [s]', 'Fontsize', 12)
        ylabel('A - acceleration [mm/s^2]', 'Fontsize', 12)
        xlim([0 6*pi])
        plot(xlim,0*[1 1], ':k', 'Handlevisibility', 'off')
        legend('show')
        
    otherwise
end

%% FUNCTION sub_solve_dyna Dynamic
function [varargout] = sub_solve_dyna(varargin)

global data...
    r l f...
    a w...
    kr kl...
    t_s u t

%tspan = 0:0.1:10; %[s]

%initial conditions
x_G1_0 = kr*sin(a);
x_G1_d_0 = kr*cos(a)*w;
y_G1_0 = kr*cos(a);
y_G1_d_0 = -kr*sin(a)*w;

b = asin((r.*sin(a)-f)/l);
b_d = (w*r*cos(a))/(l*(1-(r*sin(a)-f)^2/(l^2))^0.5);

x_G2_0 = r*sin(a)-kl*sin(b);
x_G2_d_0 = r*cos(a)*w-kl*cos(b)*b_d;
y_G2_0 = r*cos(a)+kl*cos(b);
y_G2_d_0 = -r*sin(a)*w-kl*b_d*sin(b);

%x_P_0 = r*sin(a)-kl*sin(b);
y_P_0 = r*cos(a)+l*cos(b);
y_P_d_0 = -r*sin(a)*w-l*b_d*sin(b);
u_0 = [ x_G1_0      y_G1_0      a   x_G2_0      y_G2_0      b     y_P_0       ...
       x_G1_d_0     y_G1_d_0    w   x_G2_d_0    y_G2_d_0   b_d    y_P_d_0     ...
          0      0    0   0      0     0   ];

%the function ode45 return a vector made of time instants and infos contained in a matrix u for every time instant
options = odeset('RelTol',1e-7,'AbsTol',1e-8);
disp('START dynamic compuation')
[t,u] = ode45(@sub_scm_dyna,t_s,u_0,options);
disp('END dynamic compuation')

set(data.pushbutton_solve_dyna,'Visible','off');
set(data.pushbutton_plot2,'Enable','on');
set(data.pushbutton_plot2,'Visible','on');

function u_d = sub_scm_dyna(~,u)

global data r l...
    mp mr ml...
    kr kl...
    j1 j2

a = u(3,1);
a_d = u(10,1);

b = u(6,1);
b_d = u(13,1);

% MATRICES

M = [  mr 0  0  0  0  0  0  ;...
        0  mr 0  0  0  0  0  ;...
        0  0  j1 0  0  0  0  ;...
        0  0  0 ml  0  0  0  ;...
        0  0  0  0 ml  0  0  ;...
        0  0  0  0  0 j2  0  ;...
        0  0  0  0  0  0 -mp ];

K = [   1               0               1               0               0               0       ;...
        0               1               0               1               0               0       ;...
    kr*cos(a)     -kr*sin(a)     -(r-kr)*cos(a)    (r-kr)*sin(a)        0               0       ;...
        0               0               -1              0               1               0       ;...
        0               0               0               -1              0               1       ;...
        0               0           kl*cos(b)        kl*sin(b)    (l-kl)*cos(b)  (l-kl)*sin(b)  ;...
        0               0               0               0               0               1      ];

pt_sel = get(data.popupmenu_pt,'Value');
switch pt_sel
    case 1 %F
        F = eval(get(data.Load.PT,'string'));
        T = 0;
    case 2 %T
        T = eval(get(data.Load.PT,'string'));
        F = 0;
    otherwise
end
    
fT = [  0 ;...
        0 ;...
        T ;...
        0 ;...
        0 ;...
        0 ;...
        0 ];


fF = [  0 ;...
        0 ;...
        0 ;...
        0 ;...
        0 ;...
        0 ;...
        -F ];

P = [   1   0   -kr*cos(a) 0   0   0           0  ;...
        0   1   kr*sin(a)  0   0   0           0  ;...
        0   0   -r*cos(a)  1   0   kl*cos(b)   0  ;...
        0   0   r*sin(a)   0   1   kl*sin(b)   0  ;...
        0   0   r*sin(a)   0   0   l*sin(b)    1  ;...
        0   0   r*cos(a)   0   0   -l*cos(b)   0 ];

fR = [  -((a_d)^2)*kr*sin(a) ;...
        -((a_d)^2)*kr*cos(a) ;...
        -((a_d)^2)*r*sin(a)+((b_d)^2)*kl*sin(b) ;...
        -((a_d)^2)*r*cos(a)-((b_d)^2)*kl*cos(b) ;...
        -((a_d)^2)*r*cos(a)-((b_d)^2)*l*cos(b) ;...
        ((a_d)^2)*r*sin(a)-((b_d)^2)*l*sin(b) ];

%reduction to a first order peroblem

A = [   eye(7,7)    zeros(7,7)  zeros(7,6) ;...
        zeros(7,7)  M           K ;...
        zeros(6,7)  P           zeros(6,6) ];

B = [   zeros(7,7)  -eye(7,7)   zeros(7,6) ;...
        zeros(7,7)  zeros(7,7)  zeros(7,6) ;...
        zeros(6,7)  zeros(6,7)  zeros(6,6) ];

Vec_T = [   zeros(7,1) ;...
            fT ;...
            zeros(6,1) ];

Vec_P = [   zeros(7,1) ;...
            fF ;...
            zeros(6,1) ];

Vec_R = [   zeros(7,1) ;...
            zeros(7,1) ;...
            fR ];

u_d = A\(-B*u + Vec_T + Vec_P + Vec_R);

function sub_plot2(varargin)
global data t u name_u r
%% PLOT

a = u(:,3);
a_d = u(:,10);

b = u(:,6);
b_d = u(:,13);

var = get(data.popupmenu_plot2,'value');
if var == 22 % ANIMATION
    figure (7)
    clf(7);
    PlotCrank(t,u); 
    
elseif var == 21 %FORCES RELATION
    figure (6)
    
    pt_sel = get(data.popupmenu_pt,'Value');
    switch pt_sel
        case 1 %F selected, plot T vs t
            F = eval(get(data.Load.PT,'string'));
            T = r*F.*cos((pi/2)-(a+b))./cos(b);
            
            plot(t, T,...
                '-k','LineWidth',2,...
                'Displayname', 'Torque vs Time')
            hold on
            xlabel('Time [s]', 'Fontsize', 12)
            ylabel('Torque [Nm]', 'Fontsize', 12)
            xlim([min(t) max(t)])
            %plot(xlim,zeros(1,2), ':k', 'Handlevisibility', 'off')
            legend('show')
            
        case 2 %T selected, plot F vs t
            T = eval(get(data.Load.PT,'string'));
            F = (1/r)*T.*cos((pi/2)-(a+b)).*cos(b);
            plot(t, F,...
                '-k','LineWidth',2,...
                'Displayname', 'Force vs Time')
            hold on
            xlabel('Time [s]', 'Fontsize', 12)
            ylabel('Force [N]', 'Fontsize', 12)
            xlim([min(t) max(t)])
            %plot(xlim,zeros(1,2), ':k', 'Handlevisibility', 'off')
            legend('show')
            
        otherwise
    end
    
else
    figure(5)
    set(gcf, 'Color', [1 1 1]);
    set(gcf, 'Name', [name_u{1,var},' vs Time']);
    
    plot(t, u(:,var),...
        '-k','LineWidth',2,...
        'Displayname', [name_u{1,var},' vs Time'])
    
    hold on
    
    xlabel('Time [s]', 'Fontsize', 12)
    ylabel([name_u{1,var},' ',name_u{2,var}], 'Fontsize', 12)
    xlim([min(t) max(t)])
    %plot(xlim,zeros(1,2), ':k', 'Handlevisibility', 'off')
    legend('show')
end

function PlotCrank(t,u)

global r f l

c_an = [];
g_an = [];
r_an = [];
axis equal
ylim(1.1.*[-r, r+l]);
xlim(1.1.*[min([-r,f]), max([r,f])]);
set(gcf, 'Color', [1 1 1]);
for count=1:size(t,1)
    
    a = u(count,3);
    b = u(count,6);
    x_A = r*sin(a);
    y_A = r*cos(a);
    x_G1 = u(count,1);
    y_G1 = u(count,2);
    
    x_G2 = u(count,4);
    y_G2 = u(count,5);
    x_B = f;
    y_B = u(count,7);
    
    A = [   0 x_A x_B x_G1 x_G2 ;
            0 y_A y_B y_G1 y_G2 ];
    hold on
    if isempty(c_an)
        c_an = plot(A(1,1:2),A(2,1:2),'or-','Linewidth',3); %crank
        set(0,'CurrentFigure',gcf);
    else
        try
            set(c_an,'XData',A(1,1:2),'YData',A(2,1:2));
        catch
            break;
        end
    end
    
    if isempty(r_an)
        r_an = plot(A(1,2:3),A(2,2:3),'ob-','Linewidth',3); %rod
    else
        try
            set(r_an,'XData',A(1,2:3),'YData',A(2,2:3));
        catch
            break;
        end
    end
    
    if isempty(g_an)
        g_an = plot(A(1,4:5),A(2,4:5),'og','Linewidth',3); %G1 & G2
    else
        try
            set(g_an,'XData',A(1,4:5),'YData',A(2,4:5));
        catch
            break;
        end
    end
    plot(zeros(1,2), ylim, ':k', 'Handlevisibility', 'off')
    
    %  zoom on
    %  grid on
    %  axis equal
    % axis(lim)
    %view(view_par)
    
    pause(0.05)
    drawnow
    
    %save a gif
 
    frame = getframe(gcf);
    im = frame2im(frame);
    [imind,cm] = rgb2ind(im,256,'nodither');
    if count == 1
        filename = [];
        [fname,pname] = uiputfile('.gif','Select a .gif file','animation');
        filename =[ pname, fname];
        imwrite(imind,cm,filename,'gif','Loopcount',inf,'DelayTime',0);
    else
    imwrite(imind,cm,filename,'gif','WriteMode','append','DelayTime',0);
    end
    
    
end

%% Function sub_unitselection
%% Set application data first then calling the CreateFcn.
function local_CreateFcn(h, ~, createfcn, appdata)

if ~isempty(appdata)
    names = fieldnames(appdata);
    for i=1:length(names)
        name = char(names(i));
        setappdata(h, name, getfield(appdata,name));
    end
end

if ~isempty(createfcn)
    eval(createfcn);
end
%%